function [p_val, hypo_test_text] = Unequal_var_t_test(k1, k2, se1, se2, n1, n2, varargin)

% This function performs a t-test suited for unequal variances.
% T-test formulas from [1].
% Hypothesis testing correction for multiple tests is the 
% Holm-Bonferroni correction [2].
%
% Inputs:
% k1, k2    = Values to be compared in t-test
% se1, se2  = Respective standard error of k1 and k2
% n1, n2    = Number of observations of k1 and k2
% varargin  = optional, can contain two arguments
%             First argument is significance value (defaut = 0.05)
%             Second argument is whether multipe comparisons correction of
%             [2] should be applied (default is false).
%
% Date: 08-08-2022
%
% Bibliography
% [1]   Ruxton, G. D. The unequal variance t-test is an underused
%       alternative to Students t-test and the MannWhitney U test.
%       Behavioral Ecology 17, 688690 (2006).
% [2]   Holm, S. A simple sequentially rejective multiple test procedure.
%       Scandinavian journal of statistics 6570 (1979).

switch numel(varargin)
    case 0
        alpha_value = 0.05;
        test_corr   = false(1);
    case 1
        alpha_value = varargin{1};
        test_corr   = false(1);
    case 2
        alpha_value = varargin{1};
        test_corr   = varargin{2};
end

if numel(k2) == 1 && numel(k1) > 1
    % If k2 is only 1 number and k1 is larger, compare all k1 to k2 by
    % repeating k2 to be the same size as k1.
    k2              = repmat(k2,  size(k1, 1), 1);
    se2             = repmat(se2, size(k1, 1), 1);
end
% Expand number of observations to have the same columns k1
n1                  = repmat(n1, 1, size(k1, 2));
n2                  = repmat(n2, 1, size(k1, 2));

% Formulas from [1] (t-value, degrees of freedom, and p-value)
t_val               = (k1 - k2) ./ sqrt( se1 .^ 2 + se2 .^2 );
t_dof               = (1 ./ n1 + ((se2 .^ 2) .* n2 ./ ((se1 .^ 2) .* n1)) ./ n2) .^ 2 ./ ...
                        (1 ./ ((n1 .^ 2) .* (n1 - 1)) + ((se2 .^ 2) .* n2 ./ ((se1 .^ 2) .* n1)) .^ 2  ./ ((n2 .^ 2) .* (n2 - 1)));
p_val               = 2 - 2 * tcdf(abs(t_val), t_dof);

[p_sort, ind_sort]  = sort(p_val(:));       % Sort all p-values
if test_corr
    % Each p-value has its own significance threshold, that increases as
    % the rank of the p-value increases. Once a p-value exceeds the
    % threshold, this p-value and those that are higher are accepted
    % (not-significant).
    h_rej           = logical(cumprod(p_sort < (alpha_value ./ (numel(p_val) : -1 : 1)')));
else
    % Each p-value has the same significance level
    h_rej           = p_sort < alpha_value;
end
% Set the indices of significant interactions
ind_rej             = ind_sort(h_rej);
% Default significance value
hypo_test_text      = repmat({'(n.s.)'}, size(p_val));
% Set non-significant p-values label to n.s. (not significant)
[hypo_test_text(ind_rej)] = deal({'(*)'});